/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.sis.filter;

import org.apache.sis.feature.builder.AttributeRole;
import org.apache.sis.feature.builder.FeatureTypeBuilder;

// Test dependencies
import org.junit.jupiter.api.Test;
import static org.junit.jupiter.api.Assertions.*;
import org.apache.sis.test.TestCase;
import static org.apache.sis.test.Assertions.assertSerializedEquals;

// Specific to the main branch:
import org.apache.sis.feature.AbstractFeature;
import org.apache.sis.feature.DefaultFeatureType;


/**
 * Tests {@link IdentifierFilter}.
 *
 * @author  Johann Sorel (Geomatys)
 * @author  Martin Desruisseaux (Geomatys)
 */
public final class IdentifierFilterTest extends TestCase {
    /**
     * The factory to use for creating the objects to test.
     */
    private final DefaultFilterFactory<AbstractFeature,Object,?> factory;

    /**
     * Creates a new test case.
     */
    public IdentifierFilterTest() {
        factory = DefaultFilterFactory.forFeatures();
    }

    /**
     * Tests construction and serialization.
     */
    @Test
    public void testSerialize() {
        assertSerializedEquals(factory.resourceId("abc"));
    }

    /**
     * Tests on features of diffferent types. Test data are:
     * <ul>
     *   <li>A feature type with an identifier as a string.</li>
     *   <li>A feature type with an integer identifier.</li>
     *   <li>A feature type with no identifier.</li>
     * </ul>
     */
    @Test
    public void testEvaluate() {
        final var builder = new FeatureTypeBuilder();
        builder.addAttribute(String.class).setName("att").addRole(AttributeRole.IDENTIFIER_COMPONENT);
        final AbstractFeature f1 = builder.setName("Test 1").build().newInstance();
        f1.setPropertyValue("att", "123");

        builder.clear().addAttribute(Integer.class).setName("att").addRole(AttributeRole.IDENTIFIER_COMPONENT);
        final AbstractFeature f2 = builder.setName("Test 2").build().newInstance();
        f2.setPropertyValue("att", 123);

        final AbstractFeature f3 = builder.clear().setName("Test 3").build().newInstance();

        final Filter<AbstractFeature> id = factory.resourceId("123");
        assertEquals(AbstractFeature.class, id.getResourceClass());
        assertTrue (id.test(f1));
        assertTrue (id.test(f2));
        assertFalse(id.test(f3));
    }

    /**
     * Tests evaluation of two identifiers combined by a OR logical operator.
     */
    @Test
    public void testEvaluateCombined() {
        final var builder = new FeatureTypeBuilder();
        builder.addAttribute(String.class).setName("att").addRole(AttributeRole.IDENTIFIER_COMPONENT);
        final DefaultFeatureType type = builder.setName("Test").build();

        final AbstractFeature f1 = type.newInstance(); f1.setPropertyValue("att", "123");
        final AbstractFeature f2 = type.newInstance(); f2.setPropertyValue("att", "abc");
        final AbstractFeature f3 = type.newInstance(); f3.setPropertyValue("att", "abc123");

        final Filter<AbstractFeature> id = factory.or(
                factory.resourceId("abc"),
                factory.resourceId("123"));

        assertEquals(AbstractFeature.class, id.getResourceClass());
        assertTrue (id.test(f1));
        assertTrue (id.test(f2));
        assertFalse(id.test(f3));
    }
}
