/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.store.timer;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import org.apache.rocketmq.common.MixAll;
import org.apache.rocketmq.common.UtilAll;
import org.apache.rocketmq.logging.org.slf4j.Logger;
import org.apache.rocketmq.logging.org.slf4j.LoggerFactory;
import org.apache.rocketmq.store.timer.Slot;

public class TimerWheel {
    private static final Logger log = LoggerFactory.getLogger((String)"RocketmqStore");
    public static final String TIMER_WHEEL_FILE_NAME = "timerwheel";
    public static final int BLANK = -1;
    public static final int IGNORE = -2;
    public final int slotsTotal;
    public final int precisionMs;
    private final String fileName;
    private final MappedByteBuffer mappedByteBuffer;
    private final RandomAccessFile randomAccessFile;
    private final FileChannel fileChannel;
    private final ByteBuffer byteBuffer;
    private final ThreadLocal<ByteBuffer> localBuffer = new ThreadLocal<ByteBuffer>(){

        @Override
        protected ByteBuffer initialValue() {
            return TimerWheel.this.byteBuffer.duplicate();
        }
    };
    private final int wheelLength;
    private long snapOffset;

    public TimerWheel(String fileName, int slotsTotal, int precisionMs) throws IOException {
        this(fileName, slotsTotal, precisionMs, -1L);
    }

    public TimerWheel(String fileName, int slotsTotal, int precisionMs, long snapOffset) throws IOException {
        this.slotsTotal = slotsTotal;
        this.precisionMs = precisionMs;
        this.fileName = fileName;
        this.wheelLength = this.slotsTotal * 2 * 32;
        this.snapOffset = snapOffset;
        String finalFileName = this.selectSnapshotByFlag(snapOffset);
        File file = new File(finalFileName);
        UtilAll.ensureDirOK((String)file.getParent());
        try {
            this.randomAccessFile = new RandomAccessFile(finalFileName, "rw");
            if (file.exists() && this.randomAccessFile.length() != 0L && this.randomAccessFile.length() != (long)this.wheelLength) {
                throw new RuntimeException(String.format("Timer wheel length:%d != expected:%s", this.randomAccessFile.length(), this.wheelLength));
            }
            this.randomAccessFile.setLength(this.wheelLength);
            if (snapOffset < 0L) {
                this.fileChannel = this.randomAccessFile.getChannel();
                this.mappedByteBuffer = this.fileChannel.map(FileChannel.MapMode.READ_WRITE, 0L, this.wheelLength);
                assert (this.wheelLength == this.mappedByteBuffer.remaining());
            } else {
                this.fileChannel = null;
                this.mappedByteBuffer = null;
                this.randomAccessFile.close();
            }
            this.byteBuffer = ByteBuffer.allocateDirect(this.wheelLength);
            this.byteBuffer.put(Files.readAllBytes(file.toPath()));
        }
        catch (FileNotFoundException e) {
            log.error("create file channel " + finalFileName + " Failed. ", (Throwable)e);
            throw e;
        }
        catch (IOException e) {
            log.error("map file " + finalFileName + " Failed. ", (Throwable)e);
            throw e;
        }
    }

    public void shutdown() {
        this.shutdown(true);
    }

    public void shutdown(boolean flush) {
        if (flush) {
            try {
                this.flush();
            }
            catch (Throwable e) {
                log.error("flush error when shutdown", e);
            }
        }
        UtilAll.cleanBuffer((ByteBuffer)this.mappedByteBuffer);
        UtilAll.cleanBuffer((ByteBuffer)this.byteBuffer);
        this.localBuffer.remove();
        try {
            this.fileChannel.close();
        }
        catch (Throwable t) {
            log.error("Shutdown error in timer wheel", t);
        }
    }

    public void flush() {
        if (this.mappedByteBuffer == null) {
            return;
        }
        ByteBuffer bf = this.localBuffer.get();
        bf.position(0);
        bf.limit(this.wheelLength);
        this.mappedByteBuffer.position(0);
        this.mappedByteBuffer.limit(this.wheelLength);
        for (int i = 0; i < this.wheelLength; ++i) {
            if (bf.get(i) == this.mappedByteBuffer.get(i)) continue;
            this.mappedByteBuffer.put(i, bf.get(i));
        }
        this.mappedByteBuffer.force();
    }

    public void backup(long flushWhere) throws IOException {
        ByteBuffer bf = this.localBuffer.get();
        bf.position(0);
        bf.limit(this.wheelLength);
        String fileName = this.selectSnapshotByFlag(flushWhere);
        File bakFile = new File(fileName);
        File tmpFile = new File(fileName + ".tmp");
        Files.deleteIfExists(tmpFile.toPath());
        try (RandomAccessFile randomAccessFile = new RandomAccessFile(tmpFile, "rw");
             FileChannel fileChannel = randomAccessFile.getChannel();){
            fileChannel.write(bf);
            fileChannel.force(true);
        }
        if (tmpFile.exists()) {
            Files.move(tmpFile.toPath(), bakFile.toPath(), StandardCopyOption.ATOMIC_MOVE);
            MixAll.fsyncDirectory((Path)Paths.get(bakFile.getParent(), new String[0]));
        }
        this.cleanExpiredSnapshot();
    }

    private String selectSnapshotByFlag(long flag) {
        if (flag < 0L) {
            return this.fileName;
        }
        return this.fileName + "." + flag;
    }

    public void cleanExpiredSnapshot() {
        File dir = new File(this.fileName).getParentFile();
        File[] files = dir.listFiles();
        if (files == null) {
            return;
        }
        ArrayList<FileWithFlag> snapshotFiles = new ArrayList<FileWithFlag>();
        for (File file : files) {
            long flag;
            String fileName = file.getName();
            if (!fileName.startsWith("timerwheel.") || (flag = UtilAll.asLong((String)fileName.substring(TIMER_WHEEL_FILE_NAME.length() + 1), (long)-1L)) < 0L) continue;
            snapshotFiles.add(new FileWithFlag(file, flag));
        }
        snapshotFiles.sort((a, b) -> Long.compare(b.flag, a.flag));
        for (int i = 2; i < snapshotFiles.size(); ++i) {
            UtilAll.deleteFile((File)((FileWithFlag)snapshotFiles.get((int)i)).file);
        }
    }

    public static long getMaxSnapshotFlag(String timerWheelPath) {
        File dir = new File(timerWheelPath).getParentFile();
        File[] files = dir.listFiles();
        if (files == null) {
            return -1L;
        }
        long maxFlag = -1L;
        for (File file : files) {
            long flag;
            String fileName = file.getName();
            if (!fileName.startsWith("timerwheel.") || (flag = UtilAll.asLong((String)fileName.substring(TIMER_WHEEL_FILE_NAME.length() + 1), (long)-1L)) <= maxFlag) continue;
            maxFlag = flag;
        }
        return maxFlag;
    }

    public Slot getSlot(long timeMs) {
        Slot slot = this.getRawSlot(timeMs);
        if (slot.timeMs != timeMs / (long)this.precisionMs * (long)this.precisionMs) {
            return new Slot(-1L, -1L, -1L);
        }
        return slot;
    }

    public Slot getRawSlot(long timeMs) {
        this.localBuffer.get().position(this.getSlotIndex(timeMs) * 32);
        return new Slot(this.localBuffer.get().getLong() * (long)this.precisionMs, this.localBuffer.get().getLong(), this.localBuffer.get().getLong(), this.localBuffer.get().getInt(), this.localBuffer.get().getInt());
    }

    public int getSlotIndex(long timeMs) {
        return (int)(timeMs / (long)this.precisionMs % (long)(this.slotsTotal * 2));
    }

    public void putSlot(long timeMs, long firstPos, long lastPos) {
        this.localBuffer.get().position(this.getSlotIndex(timeMs) * 32);
        this.localBuffer.get().putLong(timeMs / (long)this.precisionMs);
        this.localBuffer.get().putLong(firstPos);
        this.localBuffer.get().putLong(lastPos);
    }

    public void putSlot(long timeMs, long firstPos, long lastPos, int num, int magic) {
        this.localBuffer.get().position(this.getSlotIndex(timeMs) * 32);
        this.localBuffer.get().putLong(timeMs / (long)this.precisionMs);
        this.localBuffer.get().putLong(firstPos);
        this.localBuffer.get().putLong(lastPos);
        this.localBuffer.get().putInt(num);
        this.localBuffer.get().putInt(magic);
    }

    public void reviseSlot(long timeMs, long firstPos, long lastPos, boolean force) {
        this.localBuffer.get().position(this.getSlotIndex(timeMs) * 32);
        if (timeMs / (long)this.precisionMs != this.localBuffer.get().getLong()) {
            if (force) {
                this.putSlot(timeMs, firstPos != -2L ? firstPos : lastPos, lastPos);
            }
        } else {
            if (-2L != firstPos) {
                this.localBuffer.get().putLong(firstPos);
            } else {
                this.localBuffer.get().getLong();
            }
            if (-2L != lastPos) {
                this.localBuffer.get().putLong(lastPos);
            }
        }
    }

    public long checkPhyPos(long timeStartMs, long maxOffset) {
        long minFirst = Long.MAX_VALUE;
        int firstSlotIndex = this.getSlotIndex(timeStartMs);
        for (int i = 0; i < this.slotsTotal * 2; ++i) {
            int slotIndex = (firstSlotIndex + i) % (this.slotsTotal * 2);
            this.localBuffer.get().position(slotIndex * 32);
            if ((timeStartMs + (long)(i * this.precisionMs)) / (long)this.precisionMs != this.localBuffer.get().getLong()) continue;
            long first = this.localBuffer.get().getLong();
            long last = this.localBuffer.get().getLong();
            if (last <= maxOffset || first >= minFirst) continue;
            minFirst = first;
        }
        return minFirst;
    }

    public long getNum(long timeMs) {
        return this.getSlot((long)timeMs).num;
    }

    public long getAllNum(long timeStartMs) {
        int allNum = 0;
        int firstSlotIndex = this.getSlotIndex(timeStartMs);
        for (int i = 0; i < this.slotsTotal * 2; ++i) {
            int slotIndex = (firstSlotIndex + i) % (this.slotsTotal * 2);
            this.localBuffer.get().position(slotIndex * 32);
            if ((timeStartMs + (long)(i * this.precisionMs)) / (long)this.precisionMs != this.localBuffer.get().getLong()) continue;
            this.localBuffer.get().getLong();
            this.localBuffer.get().getLong();
            allNum += this.localBuffer.get().getInt();
        }
        return allNum;
    }

    public String getFileName() {
        return this.fileName;
    }

    private static class FileWithFlag {
        final File file;
        final long flag;

        FileWithFlag(File file, long flag) {
            this.file = file;
            this.flag = flag;
        }
    }
}

