// Copyright 2023 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.google.devtools.build.lib.packages;

import com.google.common.collect.ImmutableSet;
import com.google.devtools.build.lib.cmdline.Label;
import com.google.devtools.build.lib.vfs.Path;
import java.util.Set;
import javax.annotation.Nullable;
import net.starlark.java.syntax.Location;

/** Properties of a target needed by its parent targets. */
public interface TargetData {

  /**
   * Returns a string describing this kind of target: e.g. "cc_library rule", "source file",
   * "generated file".
   */
  String getTargetKind();

  /**
   * Returns the place where the target was defined.
   *
   * <p>The location of a rule instance is generally its "generator location", the location of the
   * outermost call on the stack, which is in the BUILD file. However, the location of a source file
   * target created by an explicit call to {@code exports_files} is the location of the innermost
   * call, which may be in an arbitrary .bzl file and is not necessarily beneath the package's
   * directory. The inconsistency seems unintentional.
   */
  Location getLocation();

  /** Returns the rule class name if the target is a rule and "" otherwise. */
  default String getRuleClass() {
    return "";
  }

  /** Returns the rule tags if the target is a rule and an empty set otherwise. */
  default Set<String> getRuleTags() {
    return ImmutableSet.of();
  }

  Label getLabel();

  default boolean isRule() {
    return false;
  }

  default boolean isFile() {
    return false;
  }

  default boolean isInputFile() {
    return false;
  }

  default boolean isOutputFile() {
    return false;
  }

  /** The generating rule's label if the target is an {@link OutputFile} otherwise null. */
  @Nullable
  default Label getGeneratingRuleLabel() {
    return null;
  }

  /** The input file path if the target is an {@link InputFile} otherwise null. */
  @Nullable
  default Path getInputPath() {
    return null;
  }

  /** Any deprecation warning of the associated rule (maybe generating) otherwise null. */
  @Nullable
  default String getDeprecationWarning() {
    return null;
  }

  /** True if the target is a testonly rule or an output file generated by a testonly rule. */
  default boolean isTestOnly() {
    return false;
  }

  /**
   * True if the underlying target advertises the required providers.
   *
   * <p>This is used to determine whether an aspect should propagate to this configured target.
   */
  default boolean satisfies(RequiredProviders required) {
    return false;
  }

  /** A test timeout if the associated rule has one. */
  @Nullable
  default TestTimeout getTestTimeout() {
    return null;
  }

  default AdvertisedProviderSet getAdvertisedProviders() {
    return AdvertisedProviderSet.EMPTY;
  }
}
