---
layout: documentation
title: Extension Overview
---

# Extension Overview

<!-- [TOC] -->

## Getting started

Bazel extensions are files ending in `.bzl`. Use a [load statement](
../build-ref.html#load) to import a symbol from an extension.

Before learning the more advanced concepts, we recommend that you first:

* Read about the [Starlark language](language.md), used in both the BUILD and
  `.bzl` files.

* Learn how you can [share variables](tutorial-sharing-variables.md) between two
  BUILD files.

## Macros and rules

A [macro](macros.md) is a function that instantiates rules. It is useful when a
BUILD file is getting too repetitive or too complex, as it allows you to reuse
some code. The function is evaluated as soon as the BUILD file is read. After
the evaluation of the BUILD file, Bazel has little information about macros: if
your macro generates a `genrule`, Bazel will behave as if you wrote the
`genrule`. As a result, `bazel query` will only list the generated `genrule`.

A [rule](rules.md) is more powerful than a macro. It can access Bazel internals
and have full control over what is going on. It may for example pass information
to other rules.

If you want to reuse simple logic, start with a macro. If a macro becomes
complex, it is often a good idea to make it a rule. Support for a new language
is typically done with a rule. Rules are for advanced users: we expect that most
people will never have to write one, they will only load and call existing
rules.

## Evaluation model

A build consists of three phases.

* **Loading phase**. First, we load and evaluate all extensions and all BUILD
  files that are needed for the build. The execution of the BUILD files simply
  instantiates rules (each time a rule is called, it gets added to a graph).
  This is where macros are evaluated.

* **Analysis phase**. The code of the rules is executed (their `implementation`
  function), and actions are instantiated. An action describes how to generate
  a set of outputs from a set of inputs, e.g. "run gcc on hello.c and get
  hello.o". It is important to note that we have to list explicitly which
  files will be generated before executing the actual commands. In other words,
  the analysis phase takes the graph generated by the loading phase and
  generates an action graph.

* **Execution phase**. Actions are executed, when at least one of their outputs is
  required. If a file is missing or if a command fails to generate one output,
  the build fails. Tests are also run during this phase.

Bazel uses parallelism to read, parse and evaluate the `.bzl` files and `BUILD`
files. A file is read at most once per build and the result of the evaluation is
cached and reused. A file is evaluated only once all its dependencies (`load()`
statements) have been resolved. By design, loading a `.bzl` file has no visible
side-effect, it only defines values and functions.

Bazel tries to be clever: it uses dependency analysis to know which files must
be loaded, which rules must be analyzed, and which actions must be executed. For
example, if a rule generates actions that we don't need for the current build,
they will not be executed.

## Creating extensions

* [Create your first macro](tutorial-creating-a-macro.md) in order to reuse some
  code. Then [learn more about macros](macros.md) and
  [using them to create "custom verbs"](tutorial-custom-verbs.md).

* [Follow the rules tutorial](rules-tutorial.md) to get started with rules.
  Next, you can read more about the [rules concepts](rules.md).

The two links below will be very useful when writing your own extensions. Keep
them within reach:

* The [API reference](lib/skylark-overview.html)

* [Examples](https://github.com/bazelbuild/examples/tree/master/rules)

## Going further

In addition to [macro](macros.md) and [rules](rules.md), you may want to write
[aspects](aspects.md) and [repository rules](repository_rules.md).

* Use [Buildifier](https://github.com/bazelbuild/buildtools) to format and lint
  your code. We recommend that you use it consistently.

* Follow the [`.bzl` style guide](bzl-style.md).

* [Test](testing.md) your code.

* [Generate documentation](https://skydoc.bazel.build/) to help your users.

* [Optimize the performance](performance.md) of your code.

* [Deploy](deploying.md) your extensions to other people.
